//+------------------------------------------------------------------+
//|                Wick Sculper V7 Rando EMA (opt-ready)             |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "7.15"
#property strict

/* Update Log:
  - V7.14: Added equity drawdown tracking + OnTester() custom score
  - V7.15: Fixed "Invalid volume" errors by normalizing lot size to broker requirements
           Added margin checking to prevent trading when account has insufficient funds
*/

//--------------------------------------------------------------------
#include <Trade/Trade.mqh>
CTrade trade;

//--- Inputs (user-tunable)
input double lot = 0.01;                // Lot size
input double eprofit = 0.10;            // Profit target in account currency
input double eloss   = 0.10;            // Loss cutoff in account currency
input bool   UseELoss = true;           // Enable/disable automatic eloss closing

input bool   OnlyOneTradeAtATime = true;// Single trade restriction
input bool   CheckMarginBeforeTrading = true; // Stop trading if insufficient margin
input double MinimumFreeMarginPercent = 100.0; // Minimum free margin % required to open trades

input int    fastMAPeriod = 14;         // Fast EMA period
input int    slowMAPeriod = 26;         // Slow EMA period

//--- Window hours (server/broker time)
input int window1Hour = 10;
input int window2Hour = 17;
input int window3Hour = 21;
input int HourOffset = 0;

//--- EMA confirmation option
input bool RequireBothConfirmations = false;

//--- Drawdown-penalty inputs
input bool UseDrawdownPenalty = true;
input double DrawdownPenalty = 0.50;

//--- Indicator handles
int fastEMAHandle = INVALID_HANDLE;
int slowEMAHandle = INVALID_HANDLE;
int m5_ema8_handle  = INVALID_HANDLE;
int m5_ema13_handle = INVALID_HANDLE;
int m5_ema21_handle = INVALID_HANDLE;
int h1_ema2_handle  = INVALID_HANDLE;
int h1_ema8_handle  = INVALID_HANDLE;

//--- EMA crossover state
bool maWasAbove = false;

//--- Track per-day trades
int lastTradeDay = -1;
bool tradedWindow[3] = {false, false, false};

//--- Magic
const int EA_MAGIC = 12345;

//--- Equity tracking
double g_initBalance = 0.0;
double g_bestEquity = 0.0;
double g_maxEquityDrawdown = 0.0;

//+------------------------------------------------------------------+
//| Normalize lot size to broker requirements                        |
//+------------------------------------------------------------------+
double NormalizeLot(double lotSize)
{
   double minLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double maxLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
   double stepLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   
   if(lotSize < minLot) lotSize = minLot;
   if(lotSize > maxLot) lotSize = maxLot;
   
   // Round to step
   lotSize = MathRound(lotSize / stepLot) * stepLot;
   
   // Normalize to proper decimal places
   int digits = (int)MathLog10(1.0 / stepLot);
   lotSize = NormalizeDouble(lotSize, digits);
   
   return lotSize;
}

//+------------------------------------------------------------------+
//| Check if there's enough margin to open a trade                   |
//+------------------------------------------------------------------+
bool HasSufficientMargin(double lotSize)
{
   if(!CheckMarginBeforeTrading) return true;
   
   double normalizedLot = NormalizeLot(lotSize);
   if(normalizedLot <= 0) return false;
   
   // Get required margin for the trade
   double requiredMargin = 0;
   if(!OrderCalcMargin(ORDER_TYPE_BUY, _Symbol, normalizedLot, 
                       SymbolInfoDouble(_Symbol, SYMBOL_ASK), requiredMargin))
   {
      Print("Failed to calculate required margin. Error: ", GetLastError());
      return false;
   }
   
   // Get current account info
   double freeMargin = AccountInfoDouble(ACCOUNT_MARGIN_FREE);
   double equity = AccountInfoDouble(ACCOUNT_EQUITY);
   
   // Calculate what free margin % would be after opening the trade
   double marginAfterTrade = freeMargin - requiredMargin;
   double freeMarginPercent = (equity > 0) ? (marginAfterTrade / equity) * 100.0 : 0;
   
   if(freeMarginPercent < MinimumFreeMarginPercent)
   {
      Print("Insufficient margin to open trade. Required: ", requiredMargin, 
            " Free: ", freeMargin, " Free% after trade: ", freeMarginPercent,
            "% (minimum required: ", MinimumFreeMarginPercent, "%)");
      return false;
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Count open positions for this EA                                 |
//+------------------------------------------------------------------+
int CountOpenPositionsForThisEA()
{
   int cnt = 0;
   int total = PositionsTotal();
   for(int i=0;i<total;i++)
   {
      ulong pos_ticket = PositionGetTicket(i);
      if(pos_ticket==0) continue;
      if(!PositionSelectByTicket(pos_ticket)) continue;
      string psym = PositionGetString(POSITION_SYMBOL);
      long pmag = (long)PositionGetInteger(POSITION_MAGIC);
      if(psym==_Symbol && pmag == EA_MAGIC) cnt++;
   }
   return cnt;
}

//+------------------------------------------------------------------+
//| Open Buy                                                          |
//+------------------------------------------------------------------+
void OpenBuy(double lotSize)
{
    if(OnlyOneTradeAtATime && CountOpenPositionsForThisEA() > 0)
    {
       Print("OpenBuy skipped: OnlyOneTradeAtATime is enabled and an open position exists.");
       return;
    }

    // Check margin before attempting trade
    if(!HasSufficientMargin(lotSize))
    {
       Print("OpenBuy skipped: Insufficient margin in account.");
       return;
    }

    // Normalize lot size
    double normalizedLot = NormalizeLot(lotSize);
    
    if(normalizedLot <= 0)
    {
       Print("OpenBuy failed: Invalid lot size after normalization. Original=", lotSize, " Normalized=", normalizedLot);
       return;
    }

    MqlTradeRequest request;
    MqlTradeResult result;
    ZeroMemory(request);
    ZeroMemory(result);

    request.action       = TRADE_ACTION_DEAL;
    request.symbol       = _Symbol;
    request.volume       = normalizedLot;
    request.type         = ORDER_TYPE_BUY;
    request.price        = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
    request.deviation    = 20;
    request.type_filling = ORDER_FILLING_FOK;
    request.magic        = EA_MAGIC;

    if (OrderSend(request, result))
    {
        if (result.deal > 0)
            Print("Buy order opened. Ticket: ", result.deal, " Volume: ", normalizedLot);
        else
            Print("OrderSend returned no deal. retcode=", result.retcode);
    }
    else
    {
        Print("OrderSend failed. retcode=", result.retcode, " Volume=", normalizedLot);
    }
}

//+------------------------------------------------------------------+
//| Open Sell                                                         |
//+------------------------------------------------------------------+
void OpenSell(double lotSize)
{
    if(OnlyOneTradeAtATime && CountOpenPositionsForThisEA() > 0)
    {
       Print("OpenSell skipped: OnlyOneTradeAtATime is enabled and an open position exists.");
       return;
    }

    // Check margin before attempting trade
    if(!HasSufficientMargin(lotSize))
    {
       Print("OpenSell skipped: Insufficient margin in account.");
       return;
    }

    // Normalize lot size
    double normalizedLot = NormalizeLot(lotSize);
    
    if(normalizedLot <= 0)
    {
       Print("OpenSell failed: Invalid lot size after normalization. Original=", lotSize, " Normalized=", normalizedLot);
       return;
    }

    MqlTradeRequest request;
    MqlTradeResult result;
    ZeroMemory(request);
    ZeroMemory(result);

    request.action       = TRADE_ACTION_DEAL;
    request.symbol       = _Symbol;
    request.volume       = normalizedLot;
    request.type         = ORDER_TYPE_SELL;
    request.price        = SymbolInfoDouble(_Symbol, SYMBOL_BID);
    request.deviation    = 20;
    request.type_filling = ORDER_FILLING_FOK;
    request.magic        = EA_MAGIC;

    if (OrderSend(request, result))
    {
        if (result.deal > 0)
            Print("Sell order opened. Ticket: ", result.deal, " Volume: ", normalizedLot);
        else
            Print("OrderSend returned no deal. retcode=", result.retcode);
    }
    else
    {
        Print("OrderSend failed. retcode=", result.retcode, " Volume=", normalizedLot);
    }
}

//+------------------------------------------------------------------+
//| Close Buy by Ticket                                              |
//+------------------------------------------------------------------+
void CloseBuyByTicket(ulong ticket)
{
    if (!PositionSelectByTicket(ticket))
    {
        Print("Failed to select position by ticket: ", ticket);
        return;
    }

    double volume = PositionGetDouble(POSITION_VOLUME);
    double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);

    MqlTradeRequest request;
    MqlTradeResult result;
    ZeroMemory(request);
    ZeroMemory(result);

    request.action       = TRADE_ACTION_DEAL;
    request.symbol       = _Symbol;
    request.volume       = volume;
    request.type         = ORDER_TYPE_SELL;
    request.price        = price;
    request.deviation    = 20;
    request.type_filling = ORDER_FILLING_FOK;
    request.position     = ticket;

    if (OrderSend(request, result))
    {
        Print("Buy position closed successfully. Ticket: ", ticket);
    }
    else
    {
        Print("Failed to close buy position. Error: ", result.retcode, " for ticket: ", ticket);
    }
}

//+------------------------------------------------------------------+
//| Close Sell by Ticket                                             |
//+------------------------------------------------------------------+
void CloseSellByTicket(ulong ticket)
{
    if (!PositionSelectByTicket(ticket))
    {
        Print("Failed to select position by ticket: ", ticket);
        return;
    }

    double volume = PositionGetDouble(POSITION_VOLUME);
    double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);

    MqlTradeRequest request;
    MqlTradeResult result;
    ZeroMemory(request);
    ZeroMemory(result);

    request.action       = TRADE_ACTION_DEAL;
    request.symbol       = _Symbol;
    request.volume       = volume;
    request.type         = ORDER_TYPE_BUY;
    request.price        = price;
    request.deviation    = 20;
    request.type_filling = ORDER_FILLING_FOK;
    request.position     = ticket;

    if (OrderSend(request, result))
    {
        Print("Sell position closed successfully. Ticket: ", ticket);
    }
    else
    {
        Print("Failed to close sell position. Error: ", result.retcode, " for ticket: ", ticket);
    }
}

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit()
{
    // Initialize equity-tracking baseline
    g_initBalance = AccountInfoDouble(ACCOUNT_BALANCE);
    g_bestEquity = AccountInfoDouble(ACCOUNT_EQUITY);
    g_maxEquityDrawdown = 0.0;

    // Print volume info for debugging
    Print("Symbol: ", _Symbol);
    Print("Min volume: ", SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN));
    Print("Max volume: ", SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX));
    Print("Volume step: ", SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP));
    Print("Input lot: ", lot, " Normalized lot: ", NormalizeLot(lot));

    // EMA crossover handles on chart timeframe
    fastEMAHandle = iMA(_Symbol, _Period, fastMAPeriod, 0, MODE_EMA, PRICE_CLOSE);
    slowEMAHandle = iMA(_Symbol, _Period, slowMAPeriod, 0, MODE_EMA, PRICE_CLOSE);

    // EMA handles for confirmations
    m5_ema8_handle  = iMA(_Symbol, PERIOD_M5, 8,  0, MODE_EMA, PRICE_CLOSE);
    m5_ema13_handle = iMA(_Symbol, PERIOD_M5, 13, 0, MODE_EMA, PRICE_CLOSE);
    m5_ema21_handle = iMA(_Symbol, PERIOD_M5, 21, 0, MODE_EMA, PRICE_CLOSE);
    h1_ema2_handle  = iMA(_Symbol, PERIOD_H1, 2, 0, MODE_EMA, PRICE_CLOSE);
    h1_ema8_handle  = iMA(_Symbol, PERIOD_H1, 8, 0, MODE_EMA, PRICE_CLOSE);

    if (fastEMAHandle == INVALID_HANDLE || slowEMAHandle == INVALID_HANDLE ||
        m5_ema8_handle == INVALID_HANDLE || m5_ema13_handle == INVALID_HANDLE || 
        m5_ema21_handle == INVALID_HANDLE || h1_ema2_handle == INVALID_HANDLE || 
        h1_ema8_handle == INVALID_HANDLE)
    {
        Print("ERROR: Failed to create one or more indicator handles. GetLastError()=", GetLastError());
        return INIT_FAILED;
    }

    // Initialize lastTradeDay and flags
    lastTradeDay = (int)(TimeCurrent() / 86400);
    for (int i = 0; i < 3; i++) tradedWindow[i] = false;

    return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    if (fastEMAHandle != INVALID_HANDLE) { IndicatorRelease(fastEMAHandle); fastEMAHandle = INVALID_HANDLE; }
    if (slowEMAHandle != INVALID_HANDLE) { IndicatorRelease(slowEMAHandle); slowEMAHandle = INVALID_HANDLE; }
    if (m5_ema8_handle  != INVALID_HANDLE) { IndicatorRelease(m5_ema8_handle);  m5_ema8_handle  = INVALID_HANDLE; }
    if (m5_ema13_handle != INVALID_HANDLE) { IndicatorRelease(m5_ema13_handle); m5_ema13_handle = INVALID_HANDLE; }
    if (m5_ema21_handle != INVALID_HANDLE) { IndicatorRelease(m5_ema21_handle); m5_ema21_handle = INVALID_HANDLE; }
    if (h1_ema2_handle != INVALID_HANDLE) { IndicatorRelease(h1_ema2_handle); h1_ema2_handle = INVALID_HANDLE; }
    if (h1_ema8_handle != INVALID_HANDLE) { IndicatorRelease(h1_ema8_handle); h1_ema8_handle = INVALID_HANDLE; }
}

//+------------------------------------------------------------------+
//| Get window index for an hour                                     |
//+------------------------------------------------------------------+
int GetWindowIndex(int hour)
{
   if (hour == (window1Hour + HourOffset)) return 0;
   if (hour == (window2Hour + HourOffset)) return 1;
   if (hour == (window3Hour + HourOffset)) return 2;
   return -1;
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    // Update equity tracking
    double curEquity = AccountInfoDouble(ACCOUNT_EQUITY);
    if(curEquity > g_bestEquity) g_bestEquity = curEquity;
    double curDD = g_bestEquity - curEquity;
    if(curDD > g_maxEquityDrawdown) g_maxEquityDrawdown = curDD;

    static datetime lastBarTime = 0;
    datetime currentBarTime = iTime(_Symbol, _Period, 0);

    // Only run logic on new candle
    if (currentBarTime != lastBarTime)
    {
        lastBarTime = currentBarTime;

        // Reset traded flags at new day
        int today = (int)(TimeCurrent() / 86400);
        if (today != lastTradeDay)
        {
            lastTradeDay = today;
            for (int r = 0; r < 3; r++) tradedWindow[r] = false;
        }

        // Determine current server hour
        datetime now = TimeCurrent();
        MqlDateTime t;
        TimeToStruct(now, t);
        int hour = t.hour;
        int winIdx = GetWindowIndex(hour);

        // Copy EMA crossover values
        double fastBuf[], slowBuf[];
        ArrayResize(fastBuf,1);
        ArrayResize(slowBuf,1);
        ArraySetAsSeries(fastBuf,false);
        ArraySetAsSeries(slowBuf,false);

        if (fastEMAHandle == INVALID_HANDLE || slowEMAHandle == INVALID_HANDLE)
        {
            Print("EMA crossover handles invalid, attempting recreate...");
            fastEMAHandle = iMA(_Symbol, _Period, fastMAPeriod, 0, MODE_EMA, PRICE_CLOSE);
            slowEMAHandle = iMA(_Symbol, _Period, slowMAPeriod, 0, MODE_EMA, PRICE_CLOSE);
            if (fastEMAHandle == INVALID_HANDLE || slowEMAHandle == INVALID_HANDLE)
            {
                Print("Failed to recreate EMA handles. Skipping bar.");
                return;
            }
        }

        int c1 = CopyBuffer(fastEMAHandle, 0, 1, 1, fastBuf);
        int c2 = CopyBuffer(slowEMAHandle, 0, 1, 1, slowBuf);
        if (c1 != 1 || c2 != 1)
        {
            Print("CopyBuffer(EMA crossover) failed. c1=", c1, " c2=", c2);
            return;
        }
        double fastVal = fastBuf[0];
        double slowVal = slowBuf[0];

        // Detect EMA crossover
        bool crossedUp = (fastVal > slowVal && !maWasAbove);
        bool crossedDown = (fastVal < slowVal && maWasAbove);

        if (crossedUp) maWasAbove = true;
        else if (crossedDown) maWasAbove = false;

        // EMA Confirmation Logic
        double m5_ema8_buf[], m5_ema13_buf[], m5_ema21_buf[], h1_ema2_buf[], h1_ema8_buf[];
        ArrayResize(m5_ema8_buf,1); ArrayResize(m5_ema13_buf,1); ArrayResize(m5_ema21_buf,1);
        ArrayResize(h1_ema2_buf,1); ArrayResize(h1_ema8_buf,1);
        ArraySetAsSeries(m5_ema8_buf,false); ArraySetAsSeries(m5_ema13_buf,false); 
        ArraySetAsSeries(m5_ema21_buf,false); ArraySetAsSeries(h1_ema2_buf,false); 
        ArraySetAsSeries(h1_ema8_buf,false);

        if (CopyBuffer(m5_ema8_handle, 0, 1, 1, m5_ema8_buf) != 1 ||
            CopyBuffer(m5_ema13_handle,0, 1, 1, m5_ema13_buf) != 1 ||
            CopyBuffer(m5_ema21_handle,0, 1, 1, m5_ema21_buf) != 1 ||
            CopyBuffer(h1_ema2_handle,  0, 1, 1, h1_ema2_buf) != 1 ||
            CopyBuffer(h1_ema8_handle,  0, 1, 1, h1_ema8_buf) != 1)
        {
            Print("CopyBuffer(EMA) failed; skipping bar. Err=", GetLastError());
            return;
        }

        double m5_ema8  = m5_ema8_buf[0];
        double m5_ema13 = m5_ema13_buf[0];
        double m5_ema21 = m5_ema21_buf[0];
        double h1_ema2  = h1_ema2_buf[0];
        double h1_ema8  = h1_ema8_buf[0];

        // M5 fan detection
        double closedClose = iClose(_Symbol, PERIOD_M5, 1);
        bool m5BullFan = (m5_ema8 > m5_ema13 && m5_ema13 > m5_ema21 && closedClose > m5_ema8);
        bool m5BearFan = (m5_ema8 < m5_ema13 && m5_ema13 < m5_ema21 && closedClose < m5_ema8);

        // H1 trend detection
        double priceH1 = iClose(_Symbol, PERIOD_H1, 1);
        bool h1Bull = (h1_ema2 > h1_ema8 && priceH1 > h1_ema2);
        bool h1Bear = (h1_ema2 < h1_ema8 && priceH1 < h1_ema2);

        // Trade if inside silver window
        if (winIdx != -1)
        {
            if (!tradedWindow[winIdx])
            {
                bool allowBuy = false;
                bool allowSell = false;

                if (crossedUp)
                {
                    bool confirmM5 = m5BullFan;
                    bool confirmH1 = h1Bull;
                    bool confirmed = RequireBothConfirmations ? (confirmM5 && confirmH1) : (confirmM5 || confirmH1);
                    if (confirmed) allowBuy = true;
                }
                else if (crossedDown)
                {
                    bool confirmM5 = m5BearFan;
                    bool confirmH1 = h1Bear;
                    bool confirmed = RequireBothConfirmations ? (confirmM5 && confirmH1) : (confirmM5 || confirmH1);
                    if (confirmed) allowSell = true;
                }

                if (allowBuy)
                {
                    Print("Silver window ", hour, ": opening BUY");
                    OpenBuy(lot);
                    tradedWindow[winIdx] = true;
                    maWasAbove = false;
                }
                else if (allowSell)
                {
                    Print("Silver window ", hour, ": opening SELL");
                    OpenSell(lot);
                    tradedWindow[winIdx] = true;
                    maWasAbove = true;
                }
            }
        }
    }

    // Check open positions for profit/loss targets
    for (int i = PositionsTotal() - 1; i >= 0; i--)
    {
        ulong ticket = PositionGetTicket(i);
        if (PositionSelectByTicket(ticket))
        {
            if (PositionGetString(POSITION_SYMBOL) == _Symbol && 
                (int)PositionGetInteger(POSITION_MAGIC) == EA_MAGIC)
            {
                double profit = PositionGetDouble(POSITION_PROFIT);

                if (profit >= eprofit)
                {
                    Print("Profit target reached. Ticket: ", ticket, " Profit=", profit);
                    if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) 
                        CloseBuyByTicket(ticket);
                    else 
                        CloseSellByTicket(ticket);
                    continue;
                }

                if (UseELoss && profit <= -eloss)
                {
                    Print("Loss threshold hit. Ticket: ", ticket, " Profit=", profit);
                    if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) 
                        CloseBuyByTicket(ticket);
                    else 
                        CloseSellByTicket(ticket);
                    continue;
                }
            }
        }
    }
}

//+------------------------------------------------------------------+
//| Tester function                                                   |
//+------------------------------------------------------------------+
double OnTester()
{
   double finalBalance = AccountInfoDouble(ACCOUNT_BALANCE);
   double netProfit = finalBalance - g_initBalance;
   double dd = g_maxEquityDrawdown;
   
   if(!UseDrawdownPenalty) return(finalBalance);
   
   double score = netProfit - DrawdownPenalty * dd;
   return(score);
}
//+------------------------------------------------------------------+